<?php
require_once '../config.php';
checkAuth();

header('Content-Type: application/json');

// Handle file uploads
function handleFileUpload($file) {
    $target_dir = "../uploads/contracts/";
    if (!file_exists($target_dir)) {
        mkdir($target_dir, 0777, true);
    }
    
    $file_extension = strtolower(pathinfo($file["name"], PATHINFO_EXTENSION));
    $new_filename = uniqid() . '.' . $file_extension;
    $target_file = $target_dir . $new_filename;
    
    if (move_uploaded_file($file["tmp_name"], $target_file)) {
        return $target_file;
    }
    return false;
}

switch ($_SERVER['REQUEST_METHOD']) {
    case 'GET':
        if (isset($_GET['id'])) {
            // Get single contract
            try {
                $stmt = $db->prepare("
                    SELECT c.*, r.start_date, r.end_date, au.name as unit_name, cu.company_name, cu.contact_name
                    FROM contracts c
                    JOIN reservations r ON c.reservation_id = r.id
                    JOIN advertising_units au ON r.unit_id = au.id
                    JOIN customers cu ON r.customer_id = cu.id
                    WHERE c.id = ?
                ");
                $stmt->execute([$_GET['id']]);
                $contract = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($contract) {
                    echo json_encode($contract);
                } else {
                    http_response_code(404);
                    echo json_encode(['error' => 'Contract not found']);
                }
            } catch(PDOException $e) {
                http_response_code(500);
                echo json_encode(['error' => $e->getMessage()]);
            }
        } else {
            // Get all contracts
            try {
                $stmt = $db->query("
                    SELECT c.*, r.start_date, r.end_date, au.name as unit_name, cu.company_name, cu.contact_name
                    FROM contracts c
                    JOIN reservations r ON c.reservation_id = r.id
                    JOIN advertising_units au ON r.unit_id = au.id
                    JOIN customers cu ON r.customer_id = cu.id
                    ORDER BY c.created_at DESC
                ");
                $contracts = $stmt->fetchAll(PDO::FETCH_ASSOC);
                echo json_encode($contracts);
            } catch(PDOException $e) {
                http_response_code(500);
                echo json_encode(['error' => $e->getMessage()]);
            }
        }
        break;
        
    case 'POST':
        try {
            if (!isset($_POST['reservation_id']) || !isset($_FILES['contract_file']) || !isset($_POST['amount'])) {
                throw new Exception('Missing required fields');
            }
            
            // Handle file upload
            $contract_file = handleFileUpload($_FILES['contract_file']);
            if (!$contract_file) {
                throw new Exception('Failed to upload contract file');
            }
            
            $db->beginTransaction();
            
            // Insert contract
            $stmt = $db->prepare("
                INSERT INTO contracts (reservation_id, contract_file, amount, payment_status, created_at)
                VALUES (?, ?, ?, 'pending', NOW())
            ");
            $stmt->execute([
                $_POST['reservation_id'],
                $contract_file,
                $_POST['amount']
            ]);
            
            $contract_id = $db->lastInsertId();
            
            $db->commit();
            
            echo json_encode(['id' => $contract_id, 'success' => true]);
        } catch(Exception $e) {
            if ($db->inTransaction()) {
                $db->rollBack();
            }
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
        break;
        
    case 'PUT':
        try {
            $data = json_decode(file_get_contents('php://input'), true);
            
            if (!isset($data['id'])) {
                throw new Exception('Contract ID is required');
            }
            
            $db->beginTransaction();
            
            // Update contract
            $stmt = $db->prepare("
                UPDATE contracts 
                SET payment_status = ?, updated_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([
                $data['payment_status'],
                $data['id']
            ]);
            
            $db->commit();
            
            echo json_encode(['success' => true]);
        } catch(Exception $e) {
            if ($db->inTransaction()) {
                $db->rollBack();
            }
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
        break;
        
    case 'DELETE':
        try {
            if (!isset($_GET['id'])) {
                throw new Exception('Contract ID is required');
            }
            
            $db->beginTransaction();
            
            // Get contract file path before deletion
            $stmt = $db->prepare("SELECT contract_file FROM contracts WHERE id = ?");
            $stmt->execute([$_GET['id']]);
            $contract = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Delete contract
            $stmt = $db->prepare("DELETE FROM contracts WHERE id = ?");
            $stmt->execute([$_GET['id']]);
            
            // Delete contract file
            if ($contract && file_exists($contract['contract_file'])) {
                unlink($contract['contract_file']);
            }
            
            $db->commit();
            
            echo json_encode(['success' => true]);
        } catch(Exception $e) {
            if ($db->inTransaction()) {
                $db->rollBack();
            }
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
        break;
} 