<?php
require_once '../config.php';
checkAuth();

header('Content-Type: application/json');

// Handle file uploads
function handleFileUpload($file) {
    $target_dir = "../uploads/invoices/";
    if (!file_exists($target_dir)) {
        mkdir($target_dir, 0777, true);
    }
    
    $file_extension = strtolower(pathinfo($file["name"], PATHINFO_EXTENSION));
    $new_filename = uniqid() . '.' . $file_extension;
    $target_file = $target_dir . $new_filename;
    
    if (move_uploaded_file($file["tmp_name"], $target_file)) {
        return $target_file;
    }
    return false;
}

switch ($_SERVER['REQUEST_METHOD']) {
    case 'GET':
        if (isset($_GET['id'])) {
            // Get single payment
            try {
                $stmt = $db->prepare("
                    SELECT p.*, c.amount as contract_amount, cu.company_name, cu.contact_name
                    FROM payments p
                    JOIN contracts c ON p.contract_id = c.id
                    JOIN reservations r ON c.reservation_id = r.id
                    JOIN customers cu ON r.customer_id = cu.id
                    WHERE p.id = ?
                ");
                $stmt->execute([$_GET['id']]);
                $payment = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($payment) {
                    echo json_encode($payment);
                } else {
                    http_response_code(404);
                    echo json_encode(['error' => 'Ödeme bulunamadı']);
                }
            } catch(PDOException $e) {
                http_response_code(500);
                echo json_encode(['error' => $e->getMessage()]);
            }
        } else {
            // Get all payments
            try {
                $stmt = $db->query("
                    SELECT p.*, c.amount as contract_amount, cu.company_name, cu.contact_name
                    FROM payments p
                    JOIN contracts c ON p.contract_id = c.id
                    JOIN reservations r ON c.reservation_id = r.id
                    JOIN customers cu ON r.customer_id = cu.id
                    ORDER BY p.payment_date DESC
                ");
                $payments = $stmt->fetchAll(PDO::FETCH_ASSOC);
                echo json_encode($payments);
            } catch(PDOException $e) {
                http_response_code(500);
                echo json_encode(['error' => $e->getMessage()]);
            }
        }
        break;
        
    case 'POST':
        try {
            if (!isset($_POST['contract_id']) || !isset($_POST['amount']) || !isset($_POST['payment_date']) || !isset($_POST['payment_method'])) {
                throw new Exception('Eksik alanlar var');
            }
            
            $db->beginTransaction();
            
            // Insert payment
            $stmt = $db->prepare("
                INSERT INTO payments (
                    contract_id, amount, payment_date, payment_method, 
                    payment_status, description, created_at
                ) VALUES (?, ?, ?, ?, 'beklemede', ?, NOW())
            ");
            
            $stmt->execute([
                $_POST['contract_id'],
                $_POST['amount'],
                $_POST['payment_date'],
                $_POST['payment_method'],
                $_POST['description'] ?? null
            ]);
            
            $payment_id = $db->lastInsertId();
            
            // Handle invoice file upload if exists
            if (isset($_FILES['invoice_file']) && $_FILES['invoice_file']['error'] == 0) {
                $invoice_file = handleFileUpload($_FILES['invoice_file']);
                if ($invoice_file) {
                    $stmt = $db->prepare("UPDATE payments SET invoice_file = ? WHERE id = ?");
                    $stmt->execute([$invoice_file, $payment_id]);
                }
            }
            
            $db->commit();
            
            echo json_encode(['id' => $payment_id, 'success' => true]);
        } catch(Exception $e) {
            if ($db->inTransaction()) {
                $db->rollBack();
            }
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
        break;
        
    case 'PUT':
        try {
            $data = json_decode(file_get_contents('php://input'), true);
            
            if (!isset($data['id'])) {
                throw new Exception('Ödeme ID\'si gerekli');
            }
            
            $db->beginTransaction();
            
            // Update payment
            $stmt = $db->prepare("
                UPDATE payments 
                SET payment_status = ?, updated_at = NOW()
                WHERE id = ?
            ");
            
            $stmt->execute([
                $data['payment_status'],
                $data['id']
            ]);
            
            $db->commit();
            
            echo json_encode(['success' => true]);
        } catch(Exception $e) {
            if ($db->inTransaction()) {
                $db->rollBack();
            }
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
        break;
        
    case 'DELETE':
        try {
            if (!isset($_GET['id'])) {
                throw new Exception('Ödeme ID\'si gerekli');
            }
            
            $db->beginTransaction();
            
            // Get payment invoice file path before deletion
            $stmt = $db->prepare("SELECT invoice_file FROM payments WHERE id = ?");
            $stmt->execute([$_GET['id']]);
            $payment = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Delete payment
            $stmt = $db->prepare("DELETE FROM payments WHERE id = ?");
            $stmt->execute([$_GET['id']]);
            
            // Delete invoice file if exists
            if ($payment && $payment['invoice_file'] && file_exists($payment['invoice_file'])) {
                unlink($payment['invoice_file']);
            }
            
            $db->commit();
            
            echo json_encode(['success' => true]);
        } catch(Exception $e) {
            if ($db->inTransaction()) {
                $db->rollBack();
            }
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
        break;
} 