<?php
require_once '../config.php';
checkAuth();

header('Content-Type: application/json');

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

switch ($method) {
    case 'GET':
        // Get all reservations or a specific reservation
        if (isset($_GET['id'])) {
            try {
                $stmt = $db->prepare("
                    SELECT r.*, au.name as unit_name, au.type as unit_type, c.company_name, c.contact_name
                    FROM reservations r
                    JOIN advertising_units au ON r.unit_id = au.id
                    JOIN customers c ON r.customer_id = c.id
                    WHERE r.id = ?
                ");
                $stmt->execute([$_GET['id']]);
                $reservation = $stmt->fetch(PDO::FETCH_ASSOC);
                echo json_encode($reservation);
            } catch(PDOException $e) {
                http_response_code(500);
                echo json_encode(['error' => $e->getMessage()]);
            }
        } else {
            try {
                $stmt = $db->query("
                    SELECT r.*, au.name as unit_name, au.type as unit_type, c.company_name, c.contact_name
                    FROM reservations r
                    JOIN advertising_units au ON r.unit_id = au.id
                    JOIN customers c ON r.customer_id = c.id
                    ORDER BY r.start_date DESC
                ");
                $reservations = $stmt->fetchAll(PDO::FETCH_ASSOC);
                echo json_encode($reservations);
            } catch(PDOException $e) {
                http_response_code(500);
                echo json_encode(['error' => $e->getMessage()]);
            }
        }
        break;

    case 'POST':
        // Create new reservation
        $data = json_decode(file_get_contents('php://input'), true);
        
        try {
            // Start transaction
            $db->beginTransaction();
            
            // Insert reservation
            $stmt = $db->prepare("
                INSERT INTO reservations (unit_id, customer_id, start_date, end_date, status)
                VALUES (?, ?, ?, ?, 'active')
            ");
            
            $stmt->execute([
                $data['unit_id'],
                $data['customer_id'],
                $data['start_date'],
                $data['end_date']
            ]);
            
            $reservation_id = $db->lastInsertId();
            
            // Update unit status
            $stmt = $db->prepare("UPDATE advertising_units SET status = 'occupied' WHERE id = ?");
            $stmt->execute([$data['unit_id']]);
            
            // Commit transaction
            $db->commit();
            
            echo json_encode(['id' => $reservation_id]);
        } catch(PDOException $e) {
            // Rollback transaction on error
            $db->rollBack();
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
        break;

    case 'PUT':
        // Update reservation
        $data = json_decode(file_get_contents('php://input'), true);
        
        try {
            // Start transaction
            $db->beginTransaction();
            
            // Get current reservation
            $stmt = $db->prepare("SELECT unit_id FROM reservations WHERE id = ?");
            $stmt->execute([$data['id']]);
            $current_reservation = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Update reservation
            $stmt = $db->prepare("
                UPDATE reservations 
                SET unit_id = ?, customer_id = ?, start_date = ?, end_date = ?, status = ?
                WHERE id = ?
            ");
            
            $stmt->execute([
                $data['unit_id'],
                $data['customer_id'],
                $data['start_date'],
                $data['end_date'],
                $data['status'] ?? 'active',
                $data['id']
            ]);
            
            // If unit changed, update old and new unit statuses
            if ($current_reservation['unit_id'] != $data['unit_id']) {
                $stmt = $db->prepare("UPDATE advertising_units SET status = 'available' WHERE id = ?");
                $stmt->execute([$current_reservation['unit_id']]);
                
                $stmt = $db->prepare("UPDATE advertising_units SET status = 'occupied' WHERE id = ?");
                $stmt->execute([$data['unit_id']]);
            }
            
            // Commit transaction
            $db->commit();
            
            echo json_encode(['success' => true]);
        } catch(PDOException $e) {
            // Rollback transaction on error
            $db->rollBack();
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
        break;

    case 'DELETE':
        // Delete reservation
        $id = $_GET['id'] ?? null;
        
        if ($id) {
            try {
                // Start transaction
                $db->beginTransaction();
                
                // Get reservation details
                $stmt = $db->prepare("SELECT unit_id FROM reservations WHERE id = ?");
                $stmt->execute([$id]);
                $reservation = $stmt->fetch(PDO::FETCH_ASSOC);
                
                // Delete reservation
                $stmt = $db->prepare("DELETE FROM reservations WHERE id = ?");
                $stmt->execute([$id]);
                
                // Update unit status
                $stmt = $db->prepare("UPDATE advertising_units SET status = 'available' WHERE id = ?");
                $stmt->execute([$reservation['unit_id']]);
                
                // Commit transaction
                $db->commit();
                
                echo json_encode(['success' => true]);
            } catch(PDOException $e) {
                // Rollback transaction on error
                $db->rollBack();
                http_response_code(500);
                echo json_encode(['error' => $e->getMessage()]);
            }
        } else {
            http_response_code(400);
            echo json_encode(['error' => 'ID parameter is required']);
        }
        break;

    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
        break;
}
?> 