<?php
require_once 'config.php';
checkAuth();

// Get all contracts with related data
try {
    $stmt = $db->query("
        SELECT c.*, r.start_date, r.end_date, au.name as unit_name, cu.company_name, cu.contact_name
        FROM contracts c
        JOIN reservations r ON c.reservation_id = r.id
        JOIN advertising_units au ON r.unit_id = au.id
        JOIN customers cu ON r.customer_id = cu.id
        ORDER BY c.created_at DESC
    ");
    $contracts = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Get active reservations without contracts
    $stmt = $db->query("
        SELECT r.*, au.name as unit_name, cu.company_name
        FROM reservations r
        JOIN advertising_units au ON r.unit_id = au.id
        JOIN customers cu ON r.customer_id = cu.id
        LEFT JOIN contracts co ON r.id = co.reservation_id
        WHERE r.status = 'active' AND co.id IS NULL
        ORDER BY r.start_date
    ");
    $available_reservations = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch(PDOException $e) {
    echo "Error: " . $e->getMessage();
    $contracts = [];
    $available_reservations = [];
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sözleşmeler - <?php echo SITE_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/boxicons@2.0.7/css/boxicons.min.css" rel="stylesheet">
    <style>
        .sidebar {
            min-height: 100vh;
            background: #343a40;
            color: white;
        }
        .nav-link {
            color: rgba(255,255,255,.8);
        }
        .nav-link:hover {
            color: white;
        }
        .card {
            border: none;
            border-radius: 10px;
            box-shadow: 0 0 10px rgba(0,0,0,.1);
        }
        .contract-card {
            transition: transform .2s;
        }
        .contract-card:hover {
            transform: translateY(-5px);
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 px-0 sidebar">
                <div class="p-3">
                    <h4 class="text-center mb-4"><?php echo SITE_NAME; ?></h4>
                    <ul class="nav flex-column">
                        <li class="nav-item">
                            <a class="nav-link" href="dashboard.php">
                                <i class='bx bxs-dashboard'></i> Dashboard
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="units.php">
                                <i class='bx bxs-map'></i> Reklam Üniteleri
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="reservations.php">
                                <i class='bx bxs-calendar'></i> Rezervasyonlar
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="customers.php">
                                <i class='bx bxs-user-detail'></i> Müşteriler
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link active" href="contracts.php">
                                <i class='bx bxs-file'></i> Sözleşmeler
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="logout.php">
                                <i class='bx bxs-log-out'></i> Çıkış
                            </a>
                        </li>
                    </ul>
                </div>
            </div>

            <!-- Main Content -->
            <div class="col-md-9 col-lg-10 px-4 py-3">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2>Sözleşmeler</h2>
                    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addContractModal">
                        <i class='bx bx-plus'></i> Yeni Sözleşme
                    </button>
                </div>

                <!-- Contracts List -->
                <div class="row">
                    <?php foreach ($contracts as $contract): ?>
                    <div class="col-md-4 mb-4">
                        <div class="card contract-card">
                            <div class="card-body">
                                <h5 class="card-title"><?php echo htmlspecialchars($contract['unit_name']); ?></h5>
                                <p class="card-text">
                                    <strong>Müşteri:</strong> <?php echo htmlspecialchars($contract['company_name']); ?><br>
                                    <strong>İletişim:</strong> <?php echo htmlspecialchars($contract['contact_name']); ?><br>
                                    <strong>Başlangıç:</strong> <?php echo date('d.m.Y', strtotime($contract['start_date'])); ?><br>
                                    <strong>Bitiş:</strong> <?php echo date('d.m.Y', strtotime($contract['end_date'])); ?><br>
                                    <strong>Tutar:</strong> <?php echo number_format($contract['amount'], 2, ',', '.'); ?> TL<br>
                                    <strong>Ödeme Durumu:</strong> 
                                    <span class="badge <?php echo $contract['payment_status'] == 'completed' ? 'bg-success' : ($contract['payment_status'] == 'partial' ? 'bg-warning' : 'bg-danger'); ?>">
                                        <?php echo $contract['payment_status'] == 'completed' ? 'Tamamlandı' : ($contract['payment_status'] == 'partial' ? 'Kısmi Ödeme' : 'Beklemede'); ?>
                                    </span>
                                </p>
                                <div class="btn-group">
                                    <a href="<?php echo htmlspecialchars($contract['contract_file']); ?>" class="btn btn-sm btn-outline-primary" target="_blank">
                                        <i class='bx bx-download'></i> İndir
                                    </a>
                                    <button class="btn btn-sm btn-outline-danger" onclick="deleteContract(<?php echo $contract['id']; ?>)">
                                        <i class='bx bx-trash'></i> Sil
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    </div>

    <!-- Add Contract Modal -->
    <div class="modal fade" id="addContractModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Yeni Sözleşme</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form id="addContractForm">
                        <div class="mb-3">
                            <label class="form-label">Rezervasyon</label>
                            <select class="form-select" name="reservation_id" required>
                                <option value="">Seçiniz</option>
                                <?php foreach ($available_reservations as $reservation): ?>
                                <option value="<?php echo $reservation['id']; ?>">
                                    <?php echo htmlspecialchars($reservation['unit_name']); ?> - 
                                    <?php echo htmlspecialchars($reservation['company_name']); ?> 
                                    (<?php echo date('d.m.Y', strtotime($reservation['start_date'])); ?> - 
                                    <?php echo date('d.m.Y', strtotime($reservation['end_date'])); ?>)
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Sözleşme Dosyası</label>
                            <input type="file" class="form-control" name="contract_file" accept=".pdf" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Tutar (TL)</label>
                            <input type="number" class="form-control" name="amount" step="0.01" required>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                    <button type="button" class="btn btn-primary" onclick="saveContract()">Kaydet</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Save contract function
        function saveContract() {
            const form = document.getElementById('addContractForm');
            const formData = new FormData(form);
            
            fetch('api/contracts.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.id) {
                    location.reload();
                } else {
                    alert('Sözleşme eklenirken bir hata oluştu.');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Sözleşme eklenirken bir hata oluştu.');
            });
        }

        // Delete contract function
        function deleteContract(id) {
            if (confirm('Bu sözleşmeyi silmek istediğinizden emin misiniz?')) {
                fetch(`api/contracts.php?id=${id}`, {
                    method: 'DELETE'
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        location.reload();
                    } else {
                        alert('Sözleşme silinirken bir hata oluştu.');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Sözleşme silinirken bir hata oluştu.');
                });
            }
        }
    </script>
</body>
</html> 